/************************************************************************************************************\

Module Name:    LServer.h

Description:

References:

    Copyright (c) 2014, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LSERVER_H
#define INC_LSERVER_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "LStatus.h"
#include "Liberatus.h"

// -----------------------------------------------------------------------------------------------------------
//                                   U S A G E
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                                   O V E R V I E W
// -----------------------------------------------------------------------------------------------------------
typedef MHANDLE     LServer_Handle;

#ifndef LCLIENTSERVERSHAREDENUMS
#define LCLIENTSERVERSHAREDENUMS

#define     LCLIENTSERVER_MAX_CLIENTS               8
#define     LCLIENTSERVER_MAX_SERVER                8
#define     LCLIENTSERVER_NO_TIMEOUT                0xFFFFFFFF

#define     PRODUCT_INFO_SIZE                       0x1000


/************************************************************************************************************\

Enum:           LClientServer_AttributeType

Description:    Enumeration of attribute types.

Comments:       None.

\************************************************************************************************************/
typedef enum {
    LClientServer_AttributeType_CREATE_CLIENT,          // Attributes structure for the client Create method.
    LClientServer_AttributeType_CREATE_SERVER,          // Attributes structure for the server Create method.
    LClientServer_AttributeType_FORCE32 = 0x7FFFFFFF,   // Dummy value to force to use 32-bits.
} LClientServer_AttributeType;

/************************************************************************************************************\

Enum:           LClientServer_SubmissionType

Description:    Enumeration of submission types.

Comments:       - The various notification types share a common Submission structure. The interpretation of
                  the content is type specific.

\************************************************************************************************************/
typedef enum
{
    LClientServer_SubmissionType_HEADER,              // Header passed to the WaitSubmission method.
    LClientServer_SubmissionType_USER         = 256,  // User specific submission. Can be used with
                                                      // SendSubmission and received from Wait
    LClientServer_SubmissionType_FORCE32      = 0x7FFFFFFF,   // Dummy value to force to use 32-bits.

} LClientServer_SubmissionType;

/************************************************************************************************************\

Enum:           LClientServer_NotificationType

Description:    Enumeration of notification types.

Comments:       - The various notification types share a common Notification structure. The interpretation of
                  the content is type specific.

\************************************************************************************************************/
typedef enum
{
    LClientServer_NotificationType_HEADER,              // Header passed to the WaitNotification method.
    LClientServer_NotificationType_ERROR,               // Generic error notification. Can be received from
    LClientServer_NotificationType_USER         = 256,  // User specific notification. Can be used with
                                                        // SendNotification and received from WaitNotification.
    LClientServer_NotificationType_FORCE32      = 0x7FFFFFFF,   // Dummy value to force to use 32-bits.

} LClientServer_NotificationType;

/************************************************************************************************************\

Enum:           LClientServer_NotificationFlags

Description:    Enumeration of notification types.

Comments:       - These flags are returned in the oHeader.uFlags member of the Notification structure.
                  The flags correspond to independent bit if a bitmask and can be set simultaneously.

\************************************************************************************************************/
typedef enum
{
    LClientServer_NotificationFlags_OVERFLOW     = 0x80,    // An overflow of the notification occurred and
                                                            //  previous notifications were missed.
} LClientServer_NotificationFlags;


/************************************************************************************************************\

Structure:      LClientServer_CreateAttributes

Description:    Client server object creation attributes.

Comments:       LClientServer_AttributeType_CREATE.

\************************************************************************************************************/
typedef struct tagLClientServer_CreateAttributes
{
    LClientServer_AttributeType eType;              // Shall be LClientServer_AttributeType_CREATE
    MUINT8                      uiServerName;        // The numerical name of the server
    MUINT8                      auiServerVersion[8]; // Return the server version numbers
} LClientServer_CreateAttributes;

/************************************************************************************************************\

Structure:      LClientServer_Notification

Description:    Client and server notification.

Comments:       - A number of notification data bytes may follow the notification structure.
                - The value oHeader.uSize shall be a multiple of 8 bytes.

\************************************************************************************************************/
typedef struct tagLClientServer_Notification
{
    LClientServer_NotificationType eType;   // One of the NotificationType_* enum.

    struct tagLClientServer_NotificationHeader
    {
        MUINT8                     uSize;    // Total size of the notification header with extra data following
                                             //  the header structure. This total size doesn't include the
                                             //  initial eType member.
        MUINT8                     uiRsvd0;  // uClient
        MUINT8                     uiFlags;  // A combination of NotificationFlags_* values.
        MUINT8                     uiRsvd1;  // uServer
        MINT32                     iError;   // Error code.
        MUINT64                    uiData;   // Single data associated with the notification.
        MUINT64                    uiTag;    // Tag associated with the notification.
    } oHeader;
} LClientServer_Notification;

/************************************************************************************************************\

Structure:      LClientServer_Submission

Description:    Client and server notification.

Comments:       - A number of notification data bytes may follow the notification structure.
                - The value oHeader.uSize shall be a multiple of 8 bytes.

\************************************************************************************************************/
typedef struct tagLClientServer_Submission
{
    LClientServer_SubmissionType   eType;   // One of the NotificationType_* enum.

    struct tagLClientServer_SubmissionHeader
    {
        MUINT8                     uSize;    // Total size of the notification header with extra data following
                                             //  the header structure. This total size doesn't include the
                                             //  initial eType member.
        MUINT8                     uiRsvd0;  // uClient.
        MUINT8                     uiFlags;  // A combination of NotificationFlags_* values.
        MUINT8                     uiRsvd1;  // uServer
        MUINT8                     uiMode;
        MUINT8                     uiParam;
        MUINT8                     uiValue;
        MUINT64                    uiData;   // Single data associated with the notification.
        MUINT64                    uiTag;    // Tag associated with the notification.
    } oHeader;
} LClientServer_Submission;

#endif //LCLIENTSERVERSHAREDENUMS

#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LServer_Create

Description:    Creates a server object.

Parameters:     IN  hDev                    Handle of the device for which to create a server object.
                IN  poAttributes            Pointer to the attribute type member of a CreateAttributes
                                            structure describing the attributes of the server object to
                                            create. NULL indicates the use of the default attributes.
                OUT phServer                Pointer to receive the handle of the server object created.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_OUT_OF_RESOURCES    No more client-server channels available.

Comments:       - This method creates a "named" server object for processing clients requests. A device
                  may have up to LCLIENT_MAX_SERVERS servers running and up to LCLIENT_MAX_CLIENTS connected
                  to those servers.
                - Multiple clients may connect to the same server.
                - The client receives the server version number in the auServerVersion member of the
                  attributes structure. The client shall use this information to adapt to the capabilities
                  of the server and establish a proper communication protocol.

\************************************************************************************************************/
LAPI LStatus LServer_Create(
                LDevice_Handle                   hDev,
                LClientServer_AttributeType*     poAttributes,
                LServer_Handle*                  phServer);

/************************************************************************************************************\

Function:       LServer_Destroy

Description:    Destroys a server object.

Parameters:     IN  hServer                 Handle of a server object.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - If a single reference remains on the server object, this method destroys the object
                  and the associated system resources. Otherwise, it performs an UnRef operation on the server
                  object.
                - Remaining submissions and notifications in the client-server queues are flushed when the
                  destroy operation takes place. An application should synchronize with the completion of the
                  last submission to a server before calling this method.

\************************************************************************************************************/
LAPI LStatus LServer_Destroy(
                LServer_Handle    hServer);

/************************************************************************************************************\

Function:       LServer_GetDefaultAttributes

Description:    Gets the default values of a given type of attributes.

Parameters:     IN  hServer                 Handle of a server object.
                OUT peAttributes            Pointer to the attribute type member of a structure of attributes.
                                            On input, the type member describes the type of the structure.
                                            On output, the remaining members of the structure are filled with
                                            the default value of the attributes.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This function provides a mechanism for retrieving an attribute structure filled with default
                  values. The application is concerned only with the subset of the members it's interested in.
                - Only one type of attribute at a time can be retrieved. Only AttributeType_CREATE is
                  currently supported.
                - If the default attributes to retrieve are associated with the method Create, it's allowed to
                  pass MNULL as the value of hServer parameter.

\************************************************************************************************************/
LAPI LStatus LServer_GetDefaultAttributes(
                LServer_Handle                  hServer,
                LClientServer_AttributeType*    peAttributes );

/************************************************************************************************************\

Function:       LServer_GetAttributes

Description:    Gets the current values of a given type of attributes.

Parameters:     IN  hServer                 Handle of a server object.
                OUT peAttributes            Pointer to the attribute type member of a structure of attributes.
                                            On input, the type member describes the type of the structure.
                                            On output, the remaining members of the structure are filled with
                                            the current value of the attributes.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This function provides a mechanism for retrieving an attribute structure filled with the
                  current values.
                - Only one type of attribute at a time can be retrieved. Only AttributeType_CREATE is
                  currently supported.

\************************************************************************************************************/
LAPI LStatus LServer_GetAttributes(
                LServer_Handle                   hServer,
                LClientServer_AttributeType*     peAttributes);

/************************************************************************************************************\

Function:       LServer_WaitSubmission

Description:    Waits for a submission from the client.

Parameters:     IN     hServer              Handle of a server object.
                IN     puiClientName        Pointer to the client name associated with the submission.
                IN OUT poSubmission         Pointer to the submission type member of a Submission structure.
                                            On input, the type member describes the type of the structure and
                                            the available size. On output, all the members of the structure are
                                            filled with the information of the submission and a new type is
                                            returned.
                IN  uiTimeout_ms            Timeout, in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - This method is used to synchronize a server CPU thread with the arrival of a submission from
                  a client. The CPU thread sleeps until a submission is received or a timeout occurs.
                - The application provides a notification structure on input where the type member shall be
                  SubmissionType_HEADER. When a submission becomes available, the corresponding submission
                  header structure is filled and the type and size members are updated to match the current
                  submission. Only the header of the current submission is retrieved from the submission queue
                  when this method is called. It gives the application the opportunity to inspect the submission
                  type and the size of the submission before it retrieves the associated data using the
                  GetSubmissionData method.
                - A timeout of 0 verifies the submission status immediately without sleeping. The error
                  LStatus_TIMEOUT is returned if a submission hasn't been received.
                - Calling this method will flush pending data associated with a previous submission that
                  wasn't retrieved by calling GetSubmissionData.
                - The application shall synchronize the access to a submission queue such that a CPU thread
                  retrieving a submission will get both the header and the data before another CPU thread is
                  allowed to access the submission queue.
                - The server receives the name of the client associated with the submission through the
                  puiClientName pointer such that it can direct notifications and completions to the
                  targetteed client.

\************************************************************************************************************/
LAPI LStatus LServer_WaitSubmission(
                LServer_Handle                  hServer,
                MUINT8*                         puiClientName,
                LClientServer_SubmissionType*   poSubmission,
                MUINT32                         uiTimeout_ms );

/************************************************************************************************************\

Function:       LServer_GetSubmissionData

Description:    Waits for a submission from the client.

Parameters:     IN  hServer                 Handle of a server object.
                IN  uiClientName            Name of the client associated with the submission.
                IN  uiSize                  Size in bytes of the output data the buffer.
                OUT puiData                 Pointer to a buffer of bytes to receive the data..

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This method retrieves the data associated with the current submission.
                - An application isn't required to retrieve all or any of the data associated with a
                  submission. A call to WaitSubmission flushes any pending data from the current submission.

\************************************************************************************************************/
LAPI LStatus LServer_GetSubmissionData(
                LServer_Handle                  hServer,
                MUINT8                          uiClientName,
                MUINT32                         uiSize,
                MUINT8*                         puiData);

/************************************************************************************************************\

Function:       LServer_GetSubmissionTag

Description:    Gets the tag of the last submission to the server.

Parameters:     IN  hServer                 Handle of a client object.
                IN  uiClientName            Name of the client associated with the submission.
                OUT puiLastSubmissionTag    Pointer to an unsigned integer to receive the value of the last
                                            submission tag.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This method is used to retrieve the tag of the last submission issued to the server.
                - A client object maintains a pseudo-timeline of the submissions flowing through its submission
                  queue. A submission tag is associated with each submission received by the client object.
                  A server object maintain a record of the last submission tag received from the client. The
                  submission tag becomes a completion tag when the SendCompletionTag method of the server
                  object is called.

\************************************************************************************************************/
LAPI LStatus LServer_GetSubmissionTag(
                LServer_Handle    hServer,
                MUINT8            uiClientName,
                MUINT64*          puiLastSubmissionTag);

/************************************************************************************************************\

Function:       LServer_GetCompletionTag

Description:    Gets the tag of the last completion received from the server.

Parameters:     IN  hServer                 Handle of a client object.
                IN  uiClientName            Name of the client associated with the submission.
                OUT puiLastCompletionTag    Pointer to an unsigned integer to receive the value of the last
                                            completion tag.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This method is used to retrieve the tag of the last submission completed by the server.

\************************************************************************************************************/
LAPI LStatus LServer_GetCompletionTag(
                LServer_Handle    hServer,
                MUINT8            uiClientName,
                MUINT64*          puiLastCompletionTag);

/************************************************************************************************************\

Function:       LServer_SendCompletionTag

Description:    Waits for the tag of a completion from the server.

Parameters:     IN  hServer                 Handle of a client object.
                IN  uiClientName            Name of the client associated with the submission.
                IN  uiCompletionTag         Completion tag.
                IN  uiTimeout_ms            Timeout in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - This method notifies the client about the completion of a submission. The completion tag
                  associated with the submission is sent to the client.
                - The timeout of the operation relates to the operation of transmitting the completion tag
                  to the client.
                - A timeout of 0 attempts to send the completion tag immediately without sleeping. The error
                  LStatus_TIMEOUT is returned if the completion tag cannot be sent immediately.
\************************************************************************************************************/
LAPI LStatus LServer_SendCompletionTag(
                LServer_Handle    hServer,
                MUINT8            uiClientName,
                MUINT64           uiCompletionTag,
                MUINT32           uiTimeout_ms);

/************************************************************************************************************\

Function:       LServer_SendNotification

Description:    Sends a notification to the client.

Parameters:     IN  hServer                 Handle of a client object.
                IN  uiClientName            Name of the client associated with the notification.
                IN  poNotification          Pointer to the notification type member of a Notification structure
                                            describing the notification to send to the client.
                IN  hDeviceThread           Handle of a device thread to send the notification.
                IN  uiTimeout_ms            Timeout, in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - This method notifies the client about some event.
                - The timeout of the operation relates to the operation of transmitting the notification to
                  the client.
                - A timeout of 0 attempts to send the completion tag immediately without sleeping. The error
                  LStatus_TIMEOUT is returned if the notification cannot be sent immediately.

\************************************************************************************************************/
LAPI LStatus LServer_SendNotification(
                LServer_Handle                      hServer,
                MUINT8                              uiClientName,
                LClientServer_NotificationType*     poNotification,
                LDeviceThread_Handle                hDeviceThread,
                MUINT32                             uiTimeout_ms);

LAPI LStatus LServer_FillSharedMemoryProductInfo(
                LServer_Handle  hServer,
                MUINT8          *puiProductInfoData,
                MUINT32         uiProductInfoSize);

/************************************************************************************************************\

Function:       LServer_GetSharedMemoryProductInfo

Description:    get product info data from sharded memory.

Parameters:     IN  hServer              Handle of a server object.
                IN  uiInputDataSize      size of puiProductInfoData array.
                OUT puiProductInfoData   array of bytes.
                OUT uiProductInfoSize    size of product info data.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments: uiProductInfoSize cannot be less then 8K.

\************************************************************************************************************/
LAPI LStatus LServer_GetSharedMemoryProductInfo(
                LServer_Handle  hServer,
                MUINT32         uiInputDataSize,
                MUINT8          *puiProductInfoData,
                MUINT32*        puiProductInfoSize);

#if defined (__cplusplus)
}
#endif

#endif  // #ifndef INC_LSERVER_H
